const { app, BrowserWindow, ipcMain, Tray, Menu, globalShortcut, screen, dialog, shell } = require('electron');
const path = require('path');
const fs = require('fs');
const https = require('https');

class Store {
    constructor(opts) {
        const userDataPath = app.getPath('userData');
        this.path = path.join(userDataPath, opts.name + '.json');
        this.data = this._parseDataFile(this.path, opts.defaults);
    }
    get(key) { return this.data[key]; }
    set(key, val) {
        this.data[key] = val;
        try { fs.writeFileSync(this.path, JSON.stringify(this.data, null, 2)); }
        catch (error) { console.error('Error saving settings:', error); }
    }
    _parseDataFile(filePath, defaults) {
        try { return JSON.parse(fs.readFileSync(filePath)); }
        catch { return defaults; }
    }
}

const store = new Store({
    name: 'mini-ai-chat-sessions-v5.0',
    defaults: {
        chatSessions: [{ id: `chat-${Date.now()}`, title: "New Chat", messages: [], attachedImage: null }],
        activeChatId: `chat-${Date.now()}`,
        settings: {
            windowOpacity: 0.95,
            tintColor: '#0a0a0f', 
            activeTabColor: '#00aaff',
            userBubbleColor: '#00aaff',
            aiBubbleColor: '#27272a',
            textModel: 'openai', 
            imageModel: 'flux',
            windowBounds: { width: 450, height: 700, x: null, y: null }
        }
    }
});

let mainWindow, handleWindow, settingsWindow, tray;

function createMainWindow() {
    let savedBounds = store.get('settings').windowBounds;
    if (!savedBounds.x || !savedBounds.y) {
        const primaryDisplay = screen.getPrimaryDisplay();
        savedBounds.x = primaryDisplay.workAreaSize.width - savedBounds.width - 20;
        savedBounds.y = 50;
    }

    mainWindow = new BrowserWindow({
        ...savedBounds,
        frame: false, transparent: true, alwaysOnTop: true,
        skipTaskbar: true, show: false,
        webPreferences: { preload: path.join(__dirname, 'preload.js') }
    });

    mainWindow.loadFile(path.join(__dirname, 'index.html'));
    mainWindow.on('ready-to-show', () => mainWindow.show());
    mainWindow.on('closed', () => { mainWindow = null; });
    
    const saveBounds = () => {
        if (!mainWindow || mainWindow.isDestroyed()) return;
        let currentSettings = store.get('settings');
        currentSettings.windowBounds = mainWindow.getBounds();
        store.set('settings', currentSettings);
    };
    mainWindow.on('resize', saveBounds);
    mainWindow.on('move', saveBounds);
}

function createHandleWindow() {
    const primaryDisplay = screen.getPrimaryDisplay();
    handleWindow = new BrowserWindow({
        x: primaryDisplay.workAreaSize.width - 40, y: Math.round(primaryDisplay.workAreaSize.height / 2) - 50,
        width: 40, height: 100, frame: false, transparent: true, alwaysOnTop: true,
        skipTaskbar: true, resizable: false, focusable: false,
        webPreferences: { preload: path.join(__dirname, 'preload.js') }
    });
    handleWindow.loadFile(path.join(__dirname, 'handle.html'));
}

function createTray() {
    const iconPath = path.join(app.getAppPath(), 'assets/tray_icon.png');
    try { tray = new Tray(iconPath); } catch (e) { console.error("Could not create tray icon.", e); return; }
    
    const contextMenu = Menu.buildFromTemplate([
        { label: 'Show/Hide Chat', click: toggleWindowVisibility },
        { label: 'Settings', click: createSettingsWindow },
        { label: 'Quit', click: () => app.quit() },
    ]);
    tray.setToolTip('Mini AI Chat');
    tray.setContextMenu(contextMenu);
    tray.on('click', toggleWindowVisibility);
}

function toggleWindowVisibility() {
    if (!mainWindow || mainWindow.isDestroyed()) {
        createMainWindow();
    } else {
        mainWindow.webContents.send('toggle-slide');
    }
}

function createSettingsWindow() {
    if (settingsWindow && !settingsWindow.isDestroyed()) {
        settingsWindow.focus();
        return;
    }
    settingsWindow = new BrowserWindow({
        width: 520, height: 600, 
        parent: mainWindow,
        modal: true, frame: false, transparent: true, show: false,
        webPreferences: { preload: path.join(__dirname, 'preload.js') }
    });
    settingsWindow.loadFile(path.join(__dirname, 'settings.html'));
    settingsWindow.once('ready-to-show', () => settingsWindow.show());
    settingsWindow.on('closed', () => { settingsWindow = null; });
}

app.whenReady().then(() => {
    createMainWindow();
    createHandleWindow();
    createTray();
    globalShortcut.register('Control+Space', toggleWindowVisibility);
});

app.on('will-quit', () => {
    globalShortcut.unregisterAll();
});

// --- IPC Handlers ---
ipcMain.on('show-settings', createSettingsWindow);
ipcMain.on('close-settings-window', () => settingsWindow?.close());
ipcMain.on('toggle-slide-main', () => mainWindow?.webContents.send('toggle-slide'));
ipcMain.on('set-window-interactivity', (event, isInteractive) => {
    if(mainWindow && !mainWindow.isDestroyed()) {
        mainWindow.setIgnoreMouseEvents(!isInteractive, { forward: true });
    }
});
ipcMain.on('open-external-link', (event, url) => {
    shell.openExternal(url);
});
ipcMain.handle('get-store-data', () => store.data);
ipcMain.on('set-store-data', (event, data) => {
    Object.keys(data).forEach(key => store.set(key, data[key]));
    [mainWindow, settingsWindow].forEach(win => {
        if(win && !win.isDestroyed()) {
            win.webContents.send('store-updated', store.data);
        }
    });
});
ipcMain.handle('open-file-dialog', async () => {
    const { canceled, filePaths } = await dialog.showOpenDialog({
        properties: ['openFile'],
        filters: [{ name: 'Images', extensions: ['png', 'jpg', 'jpeg', 'webp'] }]
    });
    if (canceled || filePaths.length === 0) return null;
    try {
        const data = fs.readFileSync(filePaths[0]);
        const mimeType = `image/${path.extname(filePaths[0]).substring(1)}`;
        return `data:${mimeType};base64,${data.toString('base64')}`;
    } catch (error) {
        console.error("Failed to read file:", error);
        return null;
    }
});
ipcMain.on('save-image-to-disk', (event, url) => {
    dialog.showSaveDialog({
        title: 'Save Image', defaultPath: `pollinations-${Date.now()}.png`,
        filters: [{ name: 'Images', extensions: ['png', 'jpg'] }]
    }).then(result => {
        if (!result.canceled && result.filePath) {
            const file = fs.createWriteStream(result.filePath);
            https.get(url, (response) => response.pipe(file));
        }
    });
});